-- ***********************************************************
-- Project:  w3d importer for Westwood 3d model format
-- version:  1.08
-- Author :  Original script by coolfile / WestWood - All major
--           work and credits belong to this benign fellow !!!
-- contact:  coolfile_cn@hotmail.com
-- Remarks:  unsupport multi-material
-- -----------------------------------------------------------
-- History: 14.02.07 -- edited + structured by ElvenRider
--          01.04.11 -- amended for BFME2 by ElvenRider and
--                      incremented version from 1.07 to 1.08
--
--          24.08.11 -- bones without vertex links were previously.
--          deleted. Nonetheless, BFME requires them for attaching
--          weapons, shields, etc.  Amended by ElvenRider.
-- ***********************************************************
--------------------------------------------------------------
------------------------------------------------ debugging ---
function GetHexValue userValue =
(
	return (bit.IntAsHex userValue)
)

function GetNumValue dataValue =
(
	return (bit.and dataValue 0x7FFFFFFF)
)

chunkSize = 0; chunkType = 0
--------------------------------------------------------------
-- --------------------------------------------- Hierarchy ---
struct HierarchyHeader
(
	version, hierName, pivotCount, centerPos
)

struct HierarchyPivot
(
	pivotName, parentID, pos, eulerAngles ,rotation
)

struct Hierarchy
(
	header, pivots
)

--------------------------------------------------------------
-- --------------------------------------------- Animation ---
struct AnimationHeader
(
	version, animName, skelName, frameCount, frameRate
)

struct CompressedAnimationHeader
(
	version, animName, skelName, frameCount, frameRate, flavor
)

struct AnimationChannel
(
	firstFrame, lastFrame, vectorLen, flags, pivotID, pad, values
)

struct AnimationTimeCodedKey
(
	keyTime, keyValue
)

struct AnimationTimeCodedChannel
(
	timeCodesCount, pivotID, vectorLen, flags, values
)

struct AnimationBitChannel
(
	firstFrame, lastFrame, flags, pivotID, defaultVal, values
)

struct Animation
(
	header, channels, bitchannels
)

--------------------------------------------------------------
-- ---------------------------------------------- Material ---
struct W3dMeshMaterialSetInfo
(
	passCount, vertMatlCount, shaderCount, TextureCount
)

struct W3dMeshShader
(
	depthCompare, depthMask, colorMask, destBlend, fogFunc, priGradient,
	secGradient, srcBlend, texturing, detailColorFunc, detailAlphaFunc,
	shaderPreset, alphaTest, postDetailColorFunc, postDetailAlphaFunc, pad
)

struct W3dMeshVertMaterialInfo
(
	attrs, ambient, diffuse, specular, emissive, shininess, opacity, translucency
)

struct W3dMeshVertMaterial
(
	vmName, vmInfo, vmArgs0, vmArgs1
)

struct W3dMeshTextureInfo
(
	attrs, animType, frameCount, frameRate
)

struct W3dMeshTexture
(
	txFileName, txInfo
)

struct W3dMeshTextureCoord
(
	u, v
)

struct W3dMeshTextureStage
(
	txIds, txCoords
)

struct W3dMeshMaterialPass
(
	vmIds, shaderIds, textureStage
)

-- ---------------------------------------------- Material ---
-- ------------------------------ new structures for BFME2 ---
struct W3dNormMapHeader
(
	NumberVal, TypeName, Reserved
)

struct W3dNormMapData
(
	MapTexture01, MapTexture02,
	AmbientTxColor, DiffuseTxColor, SpecularTxColor,
	UvwData	-- Offset position of UWV coordinates
)

struct W3dNormUvwList
(
	normUvwFound, normUvwCoords
)

--------------------------------------------------------------
-- --------------------------------------------- W3dHeader ---
struct W3dMeshHeader
(
	version, attrs, meshName, containerName,
	faceCount, vertCount, matlCount, damageStageCount,
	sortLevel, prelitVersion, futureCount, vertChannelCount,
	faceChannelCount, minCorner, maxCorner, sphCenter, sphRadius
)

struct W3dMeshFace
(
	vertIds, attrs, normal, distance
)

-- -----------------------------------------------------------
-- -------------------------------------------- W3D_Meshes ---
struct W3dMesh
(
	header, verts, normals, vertInfs, faces, shadeIds,
	matlheader, shaders, vertMatls, textures, matlPass,
	normMapData	-- Extra data structure for BFME2 textures ....
)

-- -------------------------------------------- HLOdObject ---
struct HLodHeader
(
	version, lodCount, hlodName, hierarchyName
)

struct HLodObjectArrayHeader
(
	modelCount, maxScreenSize
)

struct HLodObject
(
	pivotIndex, lodName
)

struct HLodObjectArray
(
	header, lodObjects
)

struct HLod
(
	header, lodArray
)

-- ------------------------------------------- BoundingBox ---
struct AABox
(
	version, attrs, boxName, boxColor, center, extent
)

-- -----------------------------------------------------------
-- ---------------------------------------------- ReadData ---
fn GetChunkSize chunkSize =
(
	result = bit.and chunkSize 0x7FFFFFFF
)

fn ReadLongArray fstream arrayEnd=
(
	longArray = #()
	while (ftell fstream < arrayEnd) do
	(
		longArray[longArray.count+1] = ReadLong fstream #unsigned
		-- append longArray (ReadLong fstream #unsigned)
	)
	return longArray
)

fn ReadShortArray fstream arrayEnd=
(
	shortArray = #()
	while (ftell fstream < arrayEnd) do
	(
		shortArray[shortArray.count+1] = ReadShort fstream #unsigned
	)
	return shortArray
)

fn ReadFixedString fstream =
(
	strend = (ftell fstream) + 16
	instr = ReadString fstream
	fseek fstream strend #seek_set
	result = instr
)

fn ReadLongFixedString fstream =
(
	strend = (ftell fstream) + 32
	instr = ReadString fstream
	fseek fstream strend #seek_set
	result = instr
)

fn ReadColor fstream =
(
	r = ReadByte fstream #unsigned
	g = ReadByte fstream #unsigned
	b = ReadByte fstream #unsigned
	a = ReadByte fstream #unsigned
	result = Color r g b a
	--the following code will miss one byte sometime. but why?
	--result = Color (ReadByte fstream #unsigned) (ReadByte fstream #unsigned)
	--               (ReadByte fstream #unsigned) (ReadByte fstream #unsigned)
)

fn ReadSizedString fstream varSize =
(
	strend = (ftell fstream) + varSize
	instr = ReadString fstream
	fseek fstream strend #seek_set
	result = instr
)

-- --------------------------------------------- Animation ---
-- -----------------------------------------------------------
fn ReadAnimationHeader fstream =
(
	version  = ReadLong fstream #unsigned
	animName = ReadFixedString fstream
	skelName = ReadFixedString fstream
	frameCount = ReadLong fstream #unsigned
	frameRate  = ReadLong fstream #unsigned
	result = AnimationHeader version animName skelName frameCount frameRate
	--return result
)

fn ReadAnimationChannel fstream chunkEnd =
(
	firstFrame = ReadShort fstream #unsigned
	lastFrame  = ReadShort fstream #unsigned
	vectorLen  = ReadShort fstream #unsigned
	flags   = ReadShort fstream #unsigned
	pivotID = ReadShort fstream #unsigned
	--if pivotID != 0xffffffff then pivotID += 1 -- v1.00 cancle at v1.04
	pad = ReadShort fstream #unsigned
	values = #()
	i = 1
	while (ftell fstream < chunkEnd) do
	(
		case flags of
		(
			0x0006: --ANIM_CHANNEL_Q
				values[i] = quat (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)
			default: --ANIM_CHANNEL_X/Y/Z
				values[i] = ReadFloat fstream
		)
		i += 1
	)
	result = AnimationChannel firstFrame lastFrame vectorLen flags pivotID pad values
)

fn ReadAnimationBitChannel fstream chunkEnd =
(
	firstFrame = ReadShort fstream #unsigned
	lastFrame = ReadShort fstream #unsigned
	flags = ReadShort fstream #unsigned
	pivotID = ReadShort fstream #unsigned
	--if pivotID != 0xffffffff then pivotID += 1 -- v.100 cancle at v.104
	defaultVal = ReadByte fstream #unsigned
	values = #()
	i = 1
	while (ftell fstream < chunkEnd) do
	(
		values[i] = ReadByte fstream #unsigned
		i += 1
	)
	result = AnimationBitChannel firstFrame lastFrame flags pivotID defaultVal values
)

fn ReadCompressedAnimationHeader fstream =
(
	version  = ReadLong fstream #unsigned
	animName = ReadFixedString fstream
	skelName = ReadFixedString fstream
	frameCount = ReadLong fstream #unsigned
	frameRate  = ReadShort fstream #unsigned
	flavor = ReadShort fstream #unsigned
	result = CompressedAnimationHeader version animName skelName frameCount frameRate flavor
)

fn ReadAnimationTimeCodedChannel fstream chunkEnd =
(
	timeCodesCount = ReadLong fstream #unsigned
	pivotID = ReadShort fstream #unsigned
	vectorLen = ReadByte fstream #unsigned
	flags = ReadByte fstream #unsigned
	values = #()
	i = 1
	while (ftell fstream < chunkEnd) do
	(
		tCode = ReadLong fstream #unsigned
		case flags of
		(
			0x0006: --ANIM_CHANNEL_TIMECODED_Q
				tValue = quat (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)
			default: --ANIM_CHANNEL_TIMECODED_X/Y/Z
				tValue = ReadFloat fstream
		)
		values[i] = AnimationTimeCodedKey tCode tValue
		i += 1
	)
	result = AnimationTimeCodedChannel timeCodesCount pivotID vectorLen flags values
)

fn ReadAnimationTimeCodedBitChannel fstream chunkEnd =
(
	timeCodesCount = ReadLong fstream #unsigned
	pivotID = ReadShort fstream #unsigned
	flags = ReadByte fstream #unsigned
	defaultVal = ReadByte fstream #unsigned
	values = #()
	i = 1
	while (ftell fstream < chunkEnd) do
	(
		values[i] = ReadByte fstream #unsigned
		i += 1
	)
	result = AnimationBitChannel firstFrame lastFrame flags pivotID defaultVal values
)

--------------------------------------------------------------
fn ReadAnimation fstream chunkEnd =
(
	channels = #()
	bitchannels = #()
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000201:	--W3D_CHUNK_ANIMATION_HEADER
			(
				header = ReadAnimationHeader fstream
			)
			0x00000202:	--W3D_CHUNK_ANIMATION_CHANNEL	-- channel of vectors
			(
				subChunkEnd = (ftell fstream) + chunkSize
				channels[(channels.count + 1)] = ReadAnimationChannel fstream subChunkEnd --frameCount
			)
			0x00000203:	--W3D_CHUNK_BIT_CHANNEL	-- channel of boolean values (e.g. visibility)
			(
				subChunkEnd = (ftell fstream) + chunkSize
				bitchannels[(bitchannels.count + 1)] = ReadAnimationBitChannel fstream subChunkEnd
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	result = Animation header channels bitchannels
)

--------------------------------------------------------------
fn ReadCompressedAnimation fstream chunkEnd =
(
	channels = #()
	bitchannels = #()
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000281:	--W3D_CHUNK_COMPRESSED_ANIMATION_HEADER
			(
				header = ReadCompressedAnimationHeader fstream
			)
			0x00000282:	--W3D_CHUNK_COMPRESSED_ANIMATION_CHANNEL	-- channel of vectors
			(
				subChunkEnd = (ftell fstream) + chunkSize
				if header != undefined then
					animMode = header.flavor
				else
					animMode = -1
				case animMode of
				(
					0x0: --ANIM_FLAVOR_TIMECODED
					(
						channels[(channels.count + 1)] = ReadAnimationTimeCodedChannel fstream subChunkEnd --frameCount
					)
					--0x1: --ANIM_FLAVOR_ADAPTIVE_DELTA
					--0x2: --ANIM_FLAVOR_VALID
				)
			)
			0x00000283:	--W3D_CHUNK_COMPRESSED_BIT_CHANNEL	-- channel of boolean values (e.g. visibility)
			(
				subChunkEnd = (ftell fstream) + chunkSize
				bitchannels[(bitchannels.count + 1)] = ReadAnimationTimeCodedBitChannel fstream subChunkEnd
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	result = Animation header channels bitchannels
)

-- --------------------------------------------- Hierarchy ---
-- -----------------------------------------------------------
fn ReadHierarchyHeader fstream =
(
	version    = ReadLong fstream #unsigned
	hierName   = ReadFixedString fstream
	pivotCount = ReadLong fstream #unsigned
	centerPos  = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)
	result     = HierarchyHeader version hierName pivotCount centerPos
)

fn ReadHierarchyPivots fstream pivotsEnd =
(
	pivots = #()
	while (ftell fstream < pivotsEnd) do
	(
		pivotName = ReadFixedString fstream
		parentID = ReadLong fstream #unsigned	-- 0xffffffff = root pivot; no parent
		--if parentID != 0xffffffff do parentID += 1
		pos = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	-- translation to pivot point
		eulerAngles = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	-- orientation of the pivot point
		rotation = Quat (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	-- orientation of the pivot point
		pivots[(pivots.count + 1)] = HierarchyPivot pivotName parentID pos eulerAngles rotation
	)
	result = pivots
)

fn ReadHierarchy fstream chunkEnd=
(
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)

		case chunkType of
		(
			0x00000101:	--W3D_CHUNK_HIERARCHY_HEADER
			(
				header = ReadHierarchyHeader fstream
			)
			0x00000102:	--W3D_CHUNK_PIVOTS
			(
				subChunkEnd = (ftell fstream) + chunkSize
				pivots = ReadHierarchyPivots fstream subChunkEnd
			)
			0x00000103:	--W3D_CHUNK_PIVOT_FIXUPS	-- only needed by the exporter...
			(
				--fixups[(fixups.count + 1)] = ReadHierarchyFixUps fstream
				--cracked format: there are 64 bytes in 12 dwords in 3 channels in each fixup.
				--channel X : ScaleX, Unknown, Unknown, Unknown (Long/DWord)
				--channel Y : ScaleY, Unknown, Unknown, Unknown (Long/DWord)
				--channel Z : ScaleZ, Unknown, Unknown, Unknown (Long/DWord)
				fseek fstream chunkSize #seek_cur
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	result = Hierarchy header pivots
)


-- --------------------------------------------- W3dMeshes ---
-- -----------------------------------------------------------
fn ReadMeshHeader fstream =
(
	version  = ReadLong fstream #unsigned
	attrs    = ReadLong fstream #unsigned
	meshName = ReadFixedString fstream
	containerName = ReadFixedString fstream
	-- Counts, these can be regarded as an inventory of what is to come in the file.
	faceCount     = ReadLong fstream #unsigned	--number of triangles
	vertCount     = ReadLong fstream #unsigned	--number of unique vertices
	matlCount     = ReadLong fstream #unsigned	--number of unique materials
	damageStageCount = ReadLong fstream #unsigned	--number of damage offset chunks
	sortLevel     = ReadLong fstream #unsigned	--static sorting level of this mesh
	prelitVersion = ReadLong fstream #unsigned	--mesh generated by this version of Lightmap Tool
	futureCount   = ReadLong fstream #unsigned	--future counts
	vertChannelCount = ReadLong fstream #unsigned	--bits for presence of types of per-vertex info
	faceChannelCount = ReadLong fstream #unsigned	--bits for presence of types of per-face info
	-- Bounding volumes
	minCorner = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	--Min corner of the bounding box
	maxCorner = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	--Max corner of the bounding box
	sphCenter = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	--Center of bounding sphere
	sphRadius = ReadFloat fstream	--Bounding sphere radius
	result = W3dMeshHeader version attrs meshName containerName faceCount vertCount matlCount damageStageCount sortLevel prelitVersion futureCount vertChannelCount faceChannelCount minCorner maxCorner sphCenter sphRadius
)

fn ReadMeshVertArray fstream chunkEnd =
(
	verts = #()
	while (ftell fstream < chunkEnd) do
	(
		verts[verts.count+1] = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)
	)
	return verts
)

fn ReadMeshFace fstream =
(
	vertIds = Point3 (ReadLong fstream #unsigned) (ReadLong fstream #unsigned) (ReadLong fstream #unsigned)
	attrs = ReadLong fstream #unsigned	-- attributes bits
	normal = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)	-- plane normal
	distance = ReadFloat fstream	-- plane distance
	result = W3dMeshFace vertIds attrs normal distance
)

fn ReadMeshFaceArray fstream chunkEnd =
(
	faces = #()
	while (ftell fstream < chunkEnd) do
	(
		faces[faces.count+1] = ReadMeshFace fstream
	)
	return faces
)

--replaced by ReadLongArray since v1.02
--fn ReadMeshVertShadeIds fstream chunkEnd =
--(
--	shadeIds = #()
--	while (ftell fstream < chunkEnd) do
--	(
--		shadeIds[shadeIds.count+1] = ReadLong fstream #unsigned
--	)
--	return shadeIds
--)

fn ReadMeshVertInfs fstream chunkEnd =
(
	boneIds = #()
	while (ftell fstream < chunkEnd) do
	(
		append boneIds (ReadShort fstream #unsigned)
		fseek fstream 6 #seek_cur --skip pad 6 bytes
	)
	return boneIds
)

-- --------------------------------------------- Materials ---
-- -----------------------------------------------------------
fn ReadMeshMaterialSetInfo fstream chunkEnd =
(
	passCount = ReadLong fstream #unsigned	--how many material passes this render object uses
	vertMatlCount = ReadLong fstream #unsigned	--how many vertex materials are used
	shaderCount   = ReadLong fstream #unsigned	--how many shaders are used
	textureCount  = ReadLong fstream #unsigned	--how many textures are used
	result = W3dMeshMaterialSetInfo passCount vertMatlCount shaderCount textureCount
)

fn ReadMeshShader fstream =
(
	depthCompare = ReadByte fstream #unsigned
	depthMask = ReadByte fstream #unsigned
	colorMask = ReadByte fstream #unsigned	--now obsolete and ignored
	destBlend = ReadByte fstream #unsigned
	fogFunc = ReadByte fstream #unsigned	--now obsolete and ignored
	priGradient = ReadByte fstream #unsigned
	secGradient = ReadByte fstream #unsigned
	srcBlend = ReadByte fstream #unsigned
	texturing = ReadByte fstream #unsigned
	detailColorFunc = ReadByte fstream #unsigned
	detailAlphaFunc = ReadByte fstream #unsigned
	shaderPreset = ReadByte fstream #unsigned	--now obsolete and ignored
	alphaTest = ReadByte fstream #unsigned
	postDetailColorFunc = ReadByte fstream #unsigned
	postDetailAlphaFunc = ReadByte fstream #unsigned
	pad = ReadByte fstream #unsigned
	result = W3dMeshShader depthCompare depthMask colorMask destBlend fogFunc priGradient secGradient srcBlend texturing detailColorFunc detailAlphaFunc shaderPreset alphaTest postDetailColorFunc postDetailAlphaFunc pad
)

fn ReadMeshShaderArray fstream chunkEnd =
(
	shaders = #()
	while (ftell fstream < chunkEnd) do
	(
		shaders[shaders.count+1] = ReadMeshShader fstream
	)
	return shaders
)

fn ReadMeshVertMaterialInfo fstream =
(
	attrs = ReadLong fstream #unsigned	-- bitfield for the flags defined above
	ambient = ReadColor fstream
	diffuse = ReadColor fstream
	specular = ReadColor fstream
	emissive = ReadColor fstream
	shininess = ReadFloat fstream	-- how tight the specular highlight will be, 1 - 1000 (default = 1)
	opacity = ReadFloat fstream	-- how opaque the material is, 0.0 = invisible, 1.0 = fully opaque (default = 1)
	translucency = ReadFloat fstream	-- how much light passes through the material. (default = 0)
	result = W3dMeshVertMaterialInfo attrs ambient diffuse specular emissive shininess opacity translucency
)

-- --------------------------------------------- Materials ---
fn ReadMeshVertMaterial fstream chunkEnd =
(
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x0000002C:	--W3D_CHUNK_VERTEX_MATERIAL_NAME	--vertex material name (NULL-terminated string)
			(
				vmName = ReadString fstream
			)
			0x0000002D:	--W3D_CHUNK_VERTEX_MATERIAL_INFO	--W3dVertexMaterialStruct
			(
				vmInfo = ReadMeshVertMaterialInfo fstream
			)
			0x0000002E:	--W3D_CHUNK_VERTEX_MAPPER_ARGS0	--Null-terminated string
			(
				vmArgs0 = ReadString fstream
			)
			0x0000002F:	--W3D_CHUNK_VERTEX_MAPPER_ARGS1	--Null-terminated string
			(
				vmArgs1 = ReadString fstream
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	result = W3dMeshVertMaterial vmName vmInfo vmArgs0 vmArgs1
)

fn ReadMeshVertMaterialArray fstream chunkEnd =
(
	matls = #()
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x0000002B:	--W3D_CHUNK_VERTEX_MATERIAL
			(
				subChunkEnd = (ftell fstream) + chunkSize
				matls[matls.count+1] = ReadMeshVertMaterial fstream subChunkEnd
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	return matls
)

-- -----------------------------------------------------------
-- -------------------------------------------- NormalMaps ---
fn ReadNormalMapHeader fstream =
(
	NumberVal = ReadByte fstream #unsigned
	TypeName = ReadFixedString fstream
	Reserved = ReadLong fstream #unsigned
	result = W3dNormMapHeader NumberVal TypeName Reserved
)

fn ReadNormalMapEntries fstream chunkEnd =
(
	MapTexture01 = undefined
	MapTexture02 = undefined
	AmbientTxColor = undefined
	DiffuseTxColor = undefined
	SpecularTxColor= undefined
	InfoName = undefined
	AddOffset= 0
	UvwData = 0

	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000053:	--W3D_CHUNK_NORMALMAP_FLAG3	// array of material and color values
			(
				TypeFlag = ReadLong fstream #unsigned
				TypeSize = GetChunkSize(ReadLong fstream #unsigned)
				InfoName = ReadSizedString fstream TypeSize

				case TypeFlag of
				(
					0x00000001:	--W3D_NORMTYPE_TEXTURE	// texture filename (NULL-terminated string)
					(
						ItemSize = GetChunkSize(ReadLong fstream #unsigned)
						if (InfoName == "DiffuseTexture") then
						(
							MapTexture01 = ReadSizedString fstream ItemSize
						)
						if (InfoName == "NormalMap") then
						(
							MapTexture02 = ReadSizedString fstream ItemSize
						)
					)
					0x00000002:	--W3D_NORMTYPE_BUMP		// Some information etc. ....
					(
						ReadLong fstream #unsigned
					)
					0x00000005:	--W3D_NORMTYPE_COLORS	// Some color values ....
					(
						if (InfoName == "AmbientColor") then
						(
							ambi_R = ReadFloat fstream
							ambi_G = ReadFloat fstream
							ambi_B = ReadFloat fstream
							ambi_A = ReadFloat fstream
							AmbientTxColor = color ambi_R ambi_G ambi_B ambi_A
						)
						if (InfoName == "DiffuseColor") then
						(
							diff_R = ReadFloat fstream
							diff_G = ReadFloat fstream
							diff_B = ReadFloat fstream
							diff_A = ReadFloat fstream
							DiffuseTxColor = color diff_R diff_G diff_B diff_A
						)
						if (InfoName == "SpecularColor") then
						(
							spec_R = ReadFloat fstream
							spec_G = ReadFloat fstream
							spec_B = ReadFloat fstream
							spec_A = ReadFloat fstream
							SpecularTxColor = color spec_R spec_G spec_B spec_A
						)
					)
					0x00000007:	--W3D_NORMTYPE_ALPHA		// AlphaTestEnable information etc. ....
					(
						ReadByte fstream #unsigned
						ReadLong fstream #unsigned
						ReadLong fstream #unsigned
						--AddOffset = 4
					)
				)
			)
		) -- end case
	) -- end while

	-- HACK: For some reasons the offset counts will misalign for subesequent meshes.
	-- Thus, a secondary read is required after all meshes,etc have been retrieved.
	-- Moreover, the header structure is variable and not consistant due to poor design.
	if (InfoName == "SpecularExponent") then
	(
		AddOffset = 8
	)

	if (MapTexture01 != undefined) then
	(
		UvwData = (ftell fstream) + AddOffset
	)

	result = W3dNormMapData MapTexture01 MapTexture02 AmbientTxColor DiffuseTxColor SpecularTxColor UvwData
)

-- -----------------------------------------------------------
-- -------------------------------------------- UVW_Coords ---
fn LoadW3dTextureStateCoords fstream dataOffset =
(
	local normUvwCoords = #()
	if (dataOffset != 0) then
	(
		fseek fstream dataOffset #seek_set
		CHECKTYPE = ReadLong fstream #unsigned

		--  Check for W3D_CHUNK_DCG and advance to W3D_VERTEX_CHANNEL_TEXCOORD
		if (CHECKTYPE == 0x0000003B) then (
			chunkSize = GetChunkSize(ReadLong fstream #unsigned)
			fseek fstream (chunkSize + 4) #seek_cur
		)

		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		format "### DEBUG_Mapping: Type[%]  Offset[%]\n" chunkType (ftell fstream)
		--------------------------------------------------------------------------

		--  Check for W3D_VERTEX_CHANNEL_TEXCOORD
		if chunkType == 0x00000004 then
		(
			chunkType = ReadLong fstream #unsigned
			chunkSize = GetChunkSize(ReadLong fstream #unsigned)
			subChunkEnd = (ftell fstream) + chunkSize

			--  Check for W3D_CHUNK_STAGE_TEXCOORDS
			if chunkType == 0x0000004A then
			(
				while (ftell fstream < subChunkEnd) do
				(
					append normUvwCoords (Point2 (ReadFloat fstream) (ReadFloat fstream))
				)
			)
		) -- end if
	) -- end if

	if (normUvwCoords.count > 0) then
	(
		normUvwfound = true
	)
	else
	(
		normUvwfound = false
	)

	result = W3dNormUvwList normUvwfound normUvwCoords
)

--------------------------------------------------------------
-- ---------------------------------------------- Textures ---
fn ReadMeshTextureInfo fstream=
(
	attrs = ReadShort fstream #unsigned
	animType = ReadShort fstream #unsigned
	frameCount = ReadLong fstream #unsigned
	frameRate = ReadFloat fstream
	result = W3dMeshTextureInfo attrs animType frameCount frameRate
)

fn ReadMeshTexture fstream chunkEnd=
(
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000032:	--W3D_CHUNK_TEXTURE_NAME	// texture filename (NULL-terminated string)
			(
				txFileName = ReadString fstream
			)
			0x00000033:	--W3D_CHUNK_TEXTURE_INFO	// optional W3dTextureInfoStruct
			(
				txInfo = ReadMeshTextureInfo fstream
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		) -- end case
	) -- end while
	result = W3dMeshTexture txFileName txInfo
)

fn ReadMeshTextureArray fstream chunkEnd=
(
	textures = #()
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000031:	--W3D_CHUNK_TEXTURE	// wraps a texture definition
			(
				subChunkEnd = (ftell fstream) + chunkSize
				textures[(textures.count + 1)] = ReadMeshTexture fstream subChunkEnd
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		) -- end case
	) -- end while
	return textures
)

--------------------------------------------------------------
-- ---------------------------------------------- Uvw_Maps ---
fn ReadMeshTextureCoordArray fstream chunkEnd=
(
	txCoords = #()
	while (ftell fstream < chunkEnd) do
	(
		--u = ReadFloat fstream
		--v = ReadFloat fstream
		--txCoords[txCoords.count+1] = W3dMeshTextureCoord u v
		append txCoords (Point2 (ReadFloat fstream) (ReadFloat fstream))
	)
	return txCoords
)

fn ReadMeshTextureStage fstream chunkEnd=
(
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000049:	--W3D_CHUNK_TEXTURE_IDS	// single or per-tri array of uint32 texture indices (check chunk size)
			(
				subChunkEnd = (ftell fstream) + chunkSize
				txIds = ReadLongArray fstream subChunkEnd
			)
			0x0000004A:	--W3D_CHUNK_STAGE_TEXCOORDS	// per-vertex texture coordinates (array of W3dTexCoordStruct's)
			(
				subChunkEnd = (ftell fstream) + chunkSize
				txCoords = ReadMeshTextureCoordArray fstream subChunkEnd
			)
			--0x0000004B:	--W3D_CHUNK_PER_FACE_TEXCOORD_IDS	// indices to W3D_CHUNK_STAGE_TEXCOORDS, (array of Vector3i)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		) -- end case
	) -- end while
	result = W3dMeshTextureStage txIds txCoords
)

--------------------------------------------------------------
-- --------------------------------------------- Materials ---
fn ReadMeshMaterialPass fstream chunkEnd=
(
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		case chunkType of
		(
			0x00000039:	--W3D_CHUNK_VERTEX_MATERIAL_IDS	// single or per-vertex array of uint32 vertex material indices (check chunk size)
			(
				subChunkEnd = (ftell fstream) + chunkSize
				vmIds = ReadLongArray fstream subChunkEnd
			)
			0x0000003A:	--W3D_CHUNK_SHADER_IDS	// single or per-tri array of uint32 shader indices (check chunk size)
			(
				subChunkEnd = (ftell fstream) + chunkSize
				shaderIds = ReadLongArray fstream subChunkEnd
			)
			--0x0000003B:	--W3D_CHUNK_DCG	// per-vertex diffuse color values (array of W3dRGBAStruct's)
			--0x0000003C:	--W3D_CHUNK_DIG	// per-vertex diffuse illumination values (array of W3dRGBStruct's)
			--0x0000003E:	--W3D_CHUNK_SCG	// per-vertex specular color values (array of W3dRGBStruct's)
			0x00000048:	--W3D_CHUNK_TEXTURE_STAGE	// wrapper around a texture stage.
			(
				subChunkEnd = (ftell fstream) + chunkSize
				txStage = ReadMeshTextureStage fstream subChunkEnd
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		) -- end case
	) -- end while
	result = W3dMeshMaterialPass vmIds shaderIds txStage
)

-- -------------------------------------------- W3D_Meshes ---
-- -----------------------------------------------------------
fn ReadMesh fstream chunkEnd =
(
	--BFME2: added by ElvenRider
	normMapData = W3dNormMapData MapTexture01:"" MapTexture02:"" \
		AmbientTxColor:undefined DiffuseTxColor:undefined \
		SpecularTxColor:undefined UvwData:0
	isTypeMapSection = false
	offsetMapSection = 0

	--bitchannels = #()
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		subChunkEnd = (ftell fstream) + chunkSize

		case chunkType of
		(
			0x00000002:	--W3D_CHUNK_VERTICES	--array of vertices (array of W3dVectorStruct's)
			(
				verts = ReadMeshVertArray fstream subChunkEnd
			)
			0x00000003:	--W3D_CHUNK_VERTEX_NORMALS	--array of normals (array of W3dVectorStruct's)
			(
				normals = ReadMeshVertArray fstream subChunkEnd
			)
			--0x0000000C:	--W3D_CHUNK_MESH_USER_TEXT	// Text from the MAX comment field (Null terminated string)
			0x0000000E:	--W3D_CHUNK_VERTEX_INFLUENCES	--Mesh Deformation vertex connections (array of W3dVertInfStruct's)
			(
				vertInfs = ReadMeshVertInfs fstream subChunkEnd
			)
			0x0000001F:	--W3D_CHUNK_MESH_HEADER3	--mesh header contains general info about the mesh.
			(
				header = ReadMeshHeader fstream
			)
			0x00000020:	--W3D_CHUNK_TRIANGLES	--new improved triangles chunk (array of W3dTriangleStruct's)
			(
				faces = ReadMeshFaceArray fstream subChunkEnd
			)
			0x00000022:	--W3D_CHUNK_VERTEX_SHADE_INDICES	--shade indexes for each vertex (array of uint32's)
			(
				shadeIds = ReadLongArray fstream subChunkEnd
			)
			--0x00000023:	--W3D_CHUNK_PRELIT_UNLIT				// optional unlit material chunk wrapper
			--0x00000024:	--W3D_CHUNK_PRELIT_VERTEX				// optional vertex-lit material chunk wrapper
			--0x00000025:	--W3D_CHUNK_PRELIT_LIGHTMAP_MULTI_PASS	// optional lightmapped multi-pass material chunk wrapper
			--0x00000026:	--W3D_CHUNK_PRELIT_LIGHTMAP_MULTI_TEXTURE	// optional lightmapped multi-texture material chunk wrapper
			0x00000028:	--W3D_CHUNK_MATERIAL_INFO	--materials information, pass count, etc (contains W3dMaterialInfoStruct)
			(
				matlheader = ReadMeshMaterialSetInfo fstream subChunkEnd
			)
			0x00000029:	--W3D_CHUNK_SHADERS	--shaders (array of W3dShaderStruct's)
			(
				shaders = ReadMeshShaderArray fstream subChunkEnd
			)
			0x0000002A:	--W3D_CHUNK_VERTEX_MATERIALS	--wraps the vertex materials
			(
				vertMatls = ReadMeshVertMaterialArray fstream subChunkEnd
			)
			0x00000030:	--W3D_CHUNK_TEXTURES	--wraps all of the texture info
			(
				textures = ReadMeshTextureArray fstream subChunkEnd
			)
			0x00000038:	--W3D_CHUNK_MATERIAL_PASS	--wraps the information for a single material pass
			(
				matlPass = ReadMeshMaterialPass fstream subChunkEnd
			)
			--BFME2_ADD: inserted by ElvenRider
			0x00000050:	--W3D_CHUNK_NORMALMAP_INFO
			(
				offsetMapSection = subChunkEnd
			)
			0x00000051:	--W3D_CHUNK_NORMALMAP_FLAG1
			(
				isTypeMapSection = true
			)
			0x00000052:	--W3D_CHUNK_NORMALMAP_FLAG2
			(
				if (isTypeMapSection) then
				(
					normMapHeader = ReadNormalMapHeader fstream
					normMapData = ReadNormalMapEntries fstream offsetMapSection
				)
			)
			--BFME2_END:
			--0x00000058:	--W3D_CHUNK_DEFORM	// mesh deform or 'damage' information.
			--0x00000080:	--W3D_CHUNK_PS2_SHADERS	// Shader info specific to the Playstation 2.
			--0x00000090:	--W3D_CHUNK_AABTREE	// Axis-Aligned Box Tree for hierarchical polygon culling
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)

	result = W3dMesh header verts normals vertInfs faces shadeIds matlheader shaders vertMatls textures matlPass normMapData
)

-- -----------------------------------------------------------
-- --------------------------------------------- HLod Data ---
fn ReadHLodHeader fstream =
(
	version = ReadLong fstream #unsigned
	lodCount = ReadLong fstream #unsigned
	hlodName = ReadFixedString fstream
	hierarchyName = ReadFixedString fstream	-- name of the hierarchy tree to use (\0 if none)
	result = HLodHeader version lodCount hlodName hierarchyName
)

fn ReadHLodObjectArrayHeader fstream =
(
	modelCount = ReadLong fstream #unsigned
	maxScreenSize = ReadLong fstream #unsigned	-- if model is bigger than this, switch to higher lod.
	result = HLodObjectArrayHeader modelCount maxScreenSize
)

fn ReadHLodObject fstream chunkEnd =
(
	pivotIndex = ReadLong fstream #unsigned
	lodName = ReadString fstream
	fseek fstream chunkEnd #seek_set -- skip object full name 32 bytes
	result = HLodObject pivotIndex lodName
)

fn ReadHLodObjectArray fstream chunkEnd=
(
	lodObjs = #()
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		subChunkEnd = (ftell fstream) + chunkSize
		case chunkType of
		(
			0x00000703:	--W3D_CHUNK_HLOD_SUB_OBJECT_ARRAY_HEADER	// info on the objects in this level of detail array
			(
				header = ReadHLodObjectArrayHeader fstream
			)
			0x00000704:	--W3D_CHUNK_HLOD_SUB_OBJECT	// an object in this level of detail array
			(
				lodObjs[lodObjs.count+1] = ReadHLodObject fstream subChunkEnd
			)
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	result = HLodObjectArray header lodObjs
)

fn ReadHLod fstream chunkEnd =
(
	while (ftell fstream < chunkEnd) do
	(
		chunkType = ReadLong fstream #unsigned
		chunkSize = GetChunkSize(ReadLong fstream #unsigned)
		subChunkEnd = (ftell fstream) + chunkSize

		case chunkType of
		(
			0x00000701:	--W3D_CHUNK_HLOD_HEADER	// general information such as name and version
			(
				header = ReadHLodHeader fstream
			)
			0x00000702:	--W3D_CHUNK_HLOD_LOD_ARRAY	// wrapper around the array of objects for each level of detail
			(
				lodArray = ReadHLodObjectArray fstream subChunkEnd
			)
			--0x00000705:	--W3D_CHUNK_HLOD_AGGREGATE_ARRAY	// array of aggregates, contains W3D_CHUNK_SUB_OBJECT_ARRAY_HEADER and W3D_CHUNK_SUB_OBJECT_ARRAY
			--0x00000706:	--W3D_CHUNK_HLOD_PROXY_ARRAY	// array of proxies, used for application-defined purposes, provides a name and a bone.
			default:
			(
				fseek fstream chunkSize #seek_cur
			)
		)
	)
	result = HLod header lodArray
)

-- -----------------------------------------------------------
-- ------------------------------------------- BoundingBox ---
fn ReadAABox fstream chunkEnd =
(
	version = ReadLong fstream #unsigned
	attrs = ReadLong fstream #unsigned
	boxName = ReadLongFixedString fstream
	boxColor = ReadColor fstream
	center = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)
	extent = Point3 (ReadFloat fstream) (ReadFloat fstream) (ReadFloat fstream)
	result = AABox version attrs boxName boxColor center extent
)

-- ------------------------------------------ W3D_Importer ---
-- -----------------------------------------------------------
-- utility w3dimport "Coolfile W3D Importer"
function cfW3DImporter =
(
	-- for the interface in future version
	userSeparateMaterial = false

	global anim = undefined
	global cmpAnim = undefined
	global hier = undefined
	global meshes = #()
	global MeshMaps = #()
	global hlodSet = undefined
	global pickbox = undefined

	global gmPivots = #()
	global gmMeshes = #()
	global UvWMapCreated = false
	global TextureCreated= false

	fn GetHLodObject searchName =
	(
		for i = 1 to hlodSet.lodArray.lodObjects.count do
		(
			if searchName == hlodSet.lodArray.lodObjects[i].lodName do return hlodSet.lodArray.lodObjects[i]
		)
		result = undefined
	)

	w3dfilename = getOpenFileName caption:"Import W3D (BFME)" types:"Westwood 3D (*.w3d)|*.w3d|All Files (*.*)|*.*|"
	if w3dfilename != undefined then
	(
		fstream = fopen w3dfilename "rb"  ---binary
		fseek fstream 0 #seek_end
		filesize = ftell fstream
		fseek fstream 0 #seek_set
		while (ftell fstream < filesize) do
		(
			chunkType = ReadLong fstream #unsigned
			chunkSize = GetChunkSize(ReadLong fstream #unsigned)
			chunkEnd = (ftell fstream) + chunkSize

			case chunkType of
			(
				0x00000000:	--W3D_CHUNK_MESH	-- mesh definition
				(
					meshes[meshes.count+1] = ReadMesh fstream chunkEnd
				)
				0x00000100:	--W3D_CHUNK_HIERARCHY	-- hierarchy tree definition
				(
					hier = ReadHierarchy fstream chunkEnd
				)
				0x00000200:	--W3D_CHUNK_ANIMATION	-- hierarchy animation data
				(
					anim = ReadAnimation fstream chunkEnd
				)
				0x00000280:	--W3D_CHUNK_COMPRESSED_ANIMATION	// compressed hierarchy animation data
				(
					cmpAnim = ReadCompressedAnimation fstream chunkEnd
				)
				--0x000002C0:	--W3D_CHUNK_MORPH_ANIMATION	// hierarchy morphing animation data (morphs between poses, for facial animation)
				--0x00000300:	--W3D_CHUNK_HMODEL		// blueprint for a hierarchy model
				--0x00000400:	--W3D_CHUNK_LODMODEL		// blueprint for an LOD model
				--0x00000420:	--W3D_CHUNK_COLLECTION	// collection of render object names
				--0x00000440:	--W3D_CHUNK_POINTS		// array of W3dVectorStruct's.  May appear in meshes, hmodels, lodmodels, or collections.
				--0x00000460:	--W3D_CHUNK_LIGHT		// description of a light
				--0x00000500:	--W3D_CHUNK_EMITTER	// description of a particle emitter
				--0x00000600:	--W3D_CHUNK_AGGREGATE	// description of an aggregate object
				0x00000700:	--W3D_CHUNK_HLOD			// description of an HLod object (see HLodClass)
				(
					hlodSet = ReadHLod fstream chunkEnd
				)
				0x00000740:	--W3D_CHUNK_BOX	// defines an collision box render object (W3dBoxStruct)
				(
					pickbox = ReadAABox fstream chunkEnd
				)
				--0x00000741:	--W3D_CHUNK_SPHERE
				--0x00000742:	--W3D_CHUNK_RING
				--0x00000750:	--W3D_CHUNK_NULL_OBJECT	// defines a NULL object (W3dNullObjectStruct)
				--0x00000800:	--W3D_CHUNK_LIGHTSCAPE	// wrapper for lights created with Lightscape.
				--0x00000900:	--W3D_CHUNK_DAZZLE		// wrapper for a glare object.  Creates halos and flare lines seen around a bright light source
				--0x00000A00:	--W3D_CHUNK_SOUNDROBJ	// description of a sound render object
				default:
				(
					fseek fstream chunkSize #seek_cur
				)
			)
		)
		fclose fstream

		-----------------------------------------------------------------------------
		-- TASK: if hierarchy does not exist then find the dependent skelecton file
		-----------------------------------------------------------------------------
		if hier == undefined then
		(
			if hlodSet != undefined do skelfilename = (getFilenamePath w3dfilename) + hlodSet.header.hierarchyName + ".w3d"
			if anim != undefined then
				skelfilename = (getFilenamePath w3dfilename) + anim.header.skelName + ".w3d"
			else
				if cmpAnim != undefined do skelfilename = (getFilenamePath w3dfilename) + cmpAnim.header.skelName + ".w3d"
			if (skelfilename == undefined) or (not (doesFileExist skelfilename)) then
				skelfilename = getOpenFileName caption:"Import Skelecton W3D" types:"Westwood 3D (*.w3d)|*.w3d|All Files (*.*)|*.*|"

			if skelfilename != undefined then
			(
				fstream = fopen skelfilename "rb"  ---binary
				fseek fstream 0 #seek_end
				local filesize = ftell fstream

				fseek fstream 0 #seek_set
				while (ftell fstream < filesize) do
				(
					chunkType = ReadLong fstream #unsigned
					chunkSize = GetChunkSize(ReadLong fstream #unsigned)
					chunkEnd = (ftell fstream) + chunkSize
					case chunkType of
					(
						0x00000100:	--W3D_CHUNK_HIERARCHY	// hierarchy tree definition
						(
							hier = ReadHierarchy fstream chunkEnd
						)
						default:
						(
							fseek fstream chunkSize #seek_cur
						)
					)
				)
				fclose fstream
			)
		)

		-----------------------------------------------------------------------------
		-- TASK: create pivots (i.e. bones)
		-----------------------------------------------------------------------------
		if hier != undefined then
		(
			-- to skip the ROOTTRANSFORM, i starts at 2
			for i = 2 to hier.pivots.count do
			(
				-- if hier.pivots[i].parentID < 0 do continue -- -1 means ROOTTRANSFORM
				curPivot = undefined
				if hier.pivots[i].parentID == 0 then
				(
					-- gmPivots[i-1] = bone name:hier.pivots[i].pivotName pos:hier.pivots[i].pos rotation:hier.pivots[i].rotation -- v1.03 bug
					curPivot = bone name:hier.pivots[i].pivotName
					in coordsys world curPivot.rotation = hier.pivots[i].rotation
					in coordsys world curPivot.pos = hier.pivots[i].pos
				)
				else
				(
					-- gmPivots[i-1] = bone name:hier.pivots[i].pivotName parent:gmPivots[hier.pivots[i].parentID] pos:hier.pivots[i].pos rotation:hier.pivots[i].rotation
					-- in coordsys parent gmPivots[i-1].rotation = hier.pivots[i].rotation
					-- in coordsys parent gmPivots[i-1].pos = hier.pivots[i].pos
					curPivot = bone name:hier.pivots[i].pivotName parent:gmPivots[hier.pivots[i].parentID]
					in coordsys parent curPivot.rotation = hier.pivots[i].rotation
					in coordsys parent curPivot.pos = hier.pivots[i].pos
				)
				append gmPivots curPivot
			)
		)

		-----------------------------------------------------------------------------
		-- TASK: obtain extra normal map + UVWs from the model file
		-----------------------------------------------------------------------------
		fstream = fopen w3dfilename "rb"  ---binary
		for idx = 1 to meshes.count do
		(
			format "\n>>> INFO_MeshName: [%]\n" meshes[idx].header.meshName
			---------------------------------------------------------------
			MeshMaps[idx] = LoadW3dTextureStateCoords fstream meshes[idx].normMapData.UvwData
		)
		fclose fstream
		
		-----------------------------------------------------------------------------
		-- TASK: create meshes from the model file
		-----------------------------------------------------------------------------
		for i = 1 to meshes.count do
		(
			gmVerts = meshes[i].verts
			gmFaces = #()
			for f = 1 to meshes[i].faces.count do
			(
				gmFaces[f] = meshes[i].faces[f].vertIds + 1
				-- NOTE: gMax array lower-limit is 1 while w3d's is 0
			)
			gmMeshes[i] = mesh name:meshes[i].header.meshName vertices:gmVerts faces:gmFaces

			-- create a new material or apply an existing one ....
			------------------------------------------------------
			TextureCreated = false
			if (meshes[i].vertMatls != undefined) and (meshes[i].vertMatls.count > 0) then
			(
				gmMatl = undefined
				matlName = meshes[i].vertMatls[1].vmName

				if (sceneMaterials[matlName] == undefined) then
				(
					gmMatl = standardMaterial name:matlName
					gmMatl.ambient = meshes[i].vertMatls[1].vmInfo.ambient
					gmMatl.diffuse = meshes[i].vertMatls[1].vmInfo.diffuse
					gmMatl.specular = meshes[i].vertMatls[1].vmInfo.specular
					gmMatl.glossiness = meshes[i].vertMatls[1].vmInfo.shininess * 100
					gmMatl.opacity = meshes[i].vertMatls[1].vmInfo.opacity * 100
					--gmMatl.emissive = meshes[i].vertMatls[1].vmInfo.emissive
					--gmMatl.translucency = meshes[i].vertMatls[1].vmInfo.translucency
					if (meshes[i].textures != undefined) and (meshes[i].textures.count > 0) then
					(
						gmMatl.diffuseMap = bitmaptex filename:meshes[i].textures[1].txFileName
						showTextureMap gmMatl gmMatl.diffuseMap true
						TextureCreated = true
					)
					append sceneMaterials gmMatl
				)
				else
				(
					gmMatl = sceneMaterials[matlName]
				)
				gmMeshes[i].material = gmMatl
			)
			UvWMapCreated = false

			-- mapping a texture at v1.02
			-------------------------------
			if (meshes[i].matlPass != undefined) and (meshes[i].matlPass.textureStage != undefined) and\
			   (meshes[i].matlPass.textureStage.txCoords != undefined) then
			(
				numTVerts = meshes[i].matlPass.textureStage.txCoords.count
				setNumTVerts gmMeshes[i] numTVerts
				for v = 1 to numTVerts do
				(
					txCoord = meshes[i].matlPass.textureStage.txCoords[v]
					setTVert gmMeshes[i] v (Point3 txCoord.x txCoord.y 0)
				)
				numTFaces = meshes[i].faces.count
				buildTVFaces gmMeshes[i]
				for f = 1 to numTFaces do
				(
					setTVFace gmMeshes[i] f gmFaces[f]
				)
				UvWMapCreated = true
			)

			-- BFME2: create a new material or apply an existing one ....
			-------------------------------------------------------------
			if (meshes[i].normMapData.MapTexture01 != undefined) then
			(
				if (MeshMaps[i].normUvwCoords.count > 0) and (TextureCreated == false) then
				(
					mapGxMatl = undefined
					normName = "NormalMap" + i as String
					if (sceneMaterials[normName] == undefined) then
					(
						mapGxMatl = standardMaterial name:normName
						if (meshes[i].normMapData.AmbientTxColor != undefined) then (
							mapGxMatl.ambient = meshes[i].normMapData.AmbientTxColor
						)
						if (meshes[i].normMapData.DiffuseTxColor != undefined) then (
							mapGxMatl.diffuse  = meshes[i].normMapData.DiffuseTxColor
						)
						if (meshes[i].normMapData.SpecularTxColor != undefined) then (
							mapGxMatl.specular = meshes[i].normMapData.SpecularTxColor
						)
						if (meshes[i].normMapData.MapTexture01 != undefined) then (
							mapGxMatl.diffuseMap = bitmaptex filename:meshes[i].normMapData.MapTexture01
							showTextureMap mapGxMatl mapGxMatl.diffuseMap true
						)
					--NOTE: Apparently, GMAX 1.02 cannot blend 2 texture maps ....
					--if (meshes[i].normMapData.MapTexture02 != undefined) then (
					--	mapGxMatl.specularmap = bitmaptex filename:meshes[i].normMapData.MapTexture02
					--	showTextureMap mapGxMatl mapGxMatl.specularmap true
					--)
						append sceneMaterials mapGxMatl
					)
					else
					(
						mapGxMatl = sceneMaterials[normName]
					)
					gmMeshes[i].material = mapGxMatl
				)
			)

			-- BFME2: mapping a texture at v1.08 ....
			-----------------------------------------
			if (MeshMaps[i].normUvwCoords.count > 0) and (UvWMapCreated == false) then
			(
				numTVerts = MeshMaps[i].normUvwCoords.count
				setNumTVerts gmMeshes[i] numTVerts
				for vtx = 1 to numTVerts do
				(
					txCoord = MeshMaps[i].normUvwCoords[vtx]
					setTVert gmMeshes[i] vtx (Point3 txCoord.x txCoord.y 0)
				)
				numTFaces = meshes[i].faces.count
				buildTVFaces gmMeshes[i]
				for ftx = 1 to numTFaces do
				(
					setTVFace gmMeshes[i] ftx gmFaces[ftx]
				)
			)

			-- setting vert normals at v1.03
			----------------------------------
			for v = 1 to meshes[i].normals.count do
			(
				setNormal gmMeshes[i] v meshes[i].normals[v]
			)

			------------------------------------------------------------
			-- TASK: find the existing pivot of the mesh and replace it.
			------------------------------------------------------------
			if (hlodSet != undefined) and (hier != undefined) then
			(
				meshLod = GetHLodObject (meshes[i].header.containerName + "." + meshes[i].header.meshName)
				if meshLod != undefined then
				(
					p = meshLod.pivotIndex
					if p > 0 then  -- the mesh is in ROOTTRANSFORM when p == 0
					(
						if gmPivots[p] != undefined then
						(
							if gmMeshes[i].name == gmPivots[p].name then
							(
								gmMeshes[i].parent = gmPivots[p].parent
								gmMeshes[i].rotation = gmPivots[p].rotation  --have not to swap the line of rotation and pos, why?
								gmMeshes[i].pos = gmPivots[p].pos
								for c = gmPivots[p].children.count to 1 by -1 do  --only want to rearrange the hierarchy
								(
									if(gmPivots[p].children[c] != undefined) then
									(
										gmPivots[p].children[c].parent = gmMeshes[i] --not the pivot positions of the children
									)
								)
								-- NOTE: bones without vertex links were previously deleted. Nonetheless,
								--       they are needed for weapons, shields, etc.   // v1.08
								-- disusedPivot = gmPivots[p]
								-- delete disusedPivot
								gmPivots[p] = gmMeshes[i]
							)
							else
							(
								gmMeshes[i].parent = gmPivots[p]
								gmMeshes[i].rotation = gmPivots[p].rotation  --have not to swap the line of rotation and pos, why?
								gmMeshes[i].pos = gmPivots[p].pos
							)
						)
					)
				)
			)

			-- fixing the influenced verts in infantry skin at v1.03
			-----------------------------------------------------------
			if (meshes[i].vertInfs != undefined) and (hier != undefined) then
			(
				for v = 1 to meshes[i].vertInfs.count do
				(
					p = meshes[i].vertInfs[v]
					try
					(
						if (p > 0) then
						(
							in coordsys gmPivots[p] setVert gmMeshes[i] v meshes[i].verts[v]
						)
					)
					catch
					(
						format ("\nERROR has occurred: = %  %\n" p gmMeshes[i].name)
					)
				)
				-- to show influenced verts info at v1.08
				format "\nW3DMesh: % \n" meshes[i].header.meshName
				for p = 1 to hier.pivots.count do
				(
					ivsList = #()
					for v = 1 to meshes[i].vertInfs.count do
					(
						if (meshes[i].vertInfs[v] == (p - 1)) then
						(
							append ivsList v
						)
					)
					if (ivsList.count > 1) do
					(
						last = ivsList.count
						format "Vertex_Link: [%   %] ---> [%] %\n"  ivsList[1] ivsList[last] (p - 1) hier.pivots[p].pivotName
					)
				)
			)
		)

		-----------------------------------------------------------------------------
		-- TASK: Create model animations typeof FRAME-BASED + UNCOMPRESSED ....
		-----------------------------------------------------------------------------
		if (anim != undefined) and (hier != undefined) then
		(
			if anim.header.frameCount > 1 then
				animationRange = interval 1 anim.header.frameCount -- edited at v1.07
			else
				animationRange = interval 0 1
			frameRate = anim.header.frameRate
			for i = 1 to anim.channels.count do
			(
				curChn = anim.channels[i]
				--curObj = execute ("$'" + hier.pivots[curChn.pivotID + 1].pivotName + "'")
				--at v1.02. it should be curChn.pivotID + 1 at v1.04
				curObj = gmPivots[curChn.pivotID] -- test at v1.04
				--print curObj
				--print curChn
				if curObj != undefined then
				(
					datumPos = hier.pivots[curChn.pivotID + 1].pos
					datumRot = hier.pivots[curChn.pivotID + 1].rotation
					case curChn.flags of
					(
						0x0000: --ANIM_CHANNEL_X = 0
						(
							curObj.pos.controller = linear_position()
							curKey = addNewKey curObj.pos.controller 0
							curKey.value = datumPos
							for f = curChn.firstFrame to curChn.lastFrame do
							(
								k = getKeyIndex curObj.pos.controller (f + 1)
								if k == 0 then
								(
									curKey = addNewKey curObj.pos.controller (f + 1)
									curKey.value = datumPos
								)
								else
								(
									curKey = curObj.pos.controller.keys[k]
								)
								curKey.value += [curChn.values[(f - curChn.firstFrame + 1)], 0, 0] * (inverse datumRot)
							)
							--if curChn.firstFrame == 0 then
							--(
							--	curKey = addNewKey curObj.pos.controller (curChn.lastFrame + 2)
							--)
							--else
							--(
							--	curKey = addNewKey curObj.pos.controller (curChn.firstFrame)
							--)
							--curKey.value = datumPos
						)
						0x0001: --ANIM_CHANNEL_Y = 1
						(
							curObj.pos.controller = linear_position()
							curKey = addNewKey curObj.pos.controller 0
							curKey.value = datumPos
							for f = curChn.firstFrame to curChn.lastFrame do
							(
								k = getKeyIndex curObj.pos.controller (f + 1)
								if k == 0 then
								(
									curKey = addNewKey curObj.pos.controller (f + 1)
									curKey.value = datumPos
								)
								else
								(
									curKey = curObj.pos.controller.keys[k]
								)
								curKey.value += [0, curChn.values[(f - curChn.firstFrame + 1)], 0] * (inverse datumRot)
							)
							--if curChn.firstFrame == 0 then
							--(
							--	k = getKeyIndex curObj.pos.controller (curChn.lastFrame + 1)
							--	if k == 0 then
							--	(
							--		curKey = addNewKey curObj.pos.controller (curChn.lastFrame + 1)
							--		curKey.value = datumPos
							--	)
							--)
							--else
							--(
							--	k = getKeyIndex curObj.pos.controller (curChn.firstFrame - 1)
							--	if k == 0 then
							--	(
							--		curKey = addNewKey curObj.pos.controller (curChn.firstFrame - 1)
							--		curKey.value = datumPos
							--	)
							--)
						)
						0x0002: --ANIM_CHANNEL_Z = 2
						(
							curObj.pos.controller = linear_position()
							curKey = addNewKey curObj.pos.controller 0
							curKey.value = datumPos
							for f = curChn.firstFrame to curChn.lastFrame do
							(
								k = getKeyIndex curObj.pos.controller (f + 1)
								if k == 0 then
								(
									curKey = addNewKey curObj.pos.controller (f + 1)
									curKey.value = datumPos
								)
								else
								(
									curKey = curObj.pos.controller.keys[k]
								)
								curKey.value += [0, 0, curChn.values[(f - curChn.firstFrame + 1)]] * (inverse datumRot)
							)
							--if curChn.firstFrame == 0 then
							--(
							--	k = getKeyIndex curObj.pos.controller (curChn.lastFrame + 1)
							--	if k == 0 then
							--	(
							--		curKey = addNewKey curObj.pos.controller (curChn.lastFrame + 1)
							--		curKey.value = datumPos
							--	)
							--)
							--else
							--(
							--	k = getKeyIndex curObj.pos.controller (curChn.firstFrame - 1)
							--	if k == 0 then
							--	(
							--		curKey = addNewKey curObj.pos.controller (curChn.firstFrame - 1)
							--		curKey.value = datumPos
							--	)
							--)
						)
					--	--0x0003: --ANIM_CHANNEL_XR = 3
					--	--(
					--	--	curKey = addNewKey curObj.pos.controller f
					--	--	curKey.value.x = curChn.values[(f - curChn.firstFrame + 1)]
					--	--)
					--	--0x0004: --ANIM_CHANNEL_YR = 4
					--	--(
					--	--	curKey = addNewKey curObj.pos.controller f
					--	--	curKey.value.y = curChn.values[(f - curChn.firstFrame + 1)]
					--	--)
					--	--0x0005: --ANIM_CHANNEL_ZR = 5
					--	--(
					--	--	curKey = addNewKey curObj.pos.controller f
					--	--	curKey.value.z = curChn.values[(f - curChn.firstFrame + 1)]
					--	--)
						0x0006: --ANIM_CHANNEL_Q = 6
						(
							curObj.rotation.controller = linear_rotation()
							curKey = addNewKey curObj.rotation.controller 0
							curKey.value = datumRot
							for f = curChn.firstFrame to curChn.lastFrame do
							(
								curKey = addNewKey curObj.rotation.controller (f + 1)
								curKey.value = curChn.values[(f - curChn.firstFrame + 1)] - (inverse datumRot)
							)
							if curChn.firstFrame == 0 then
							(
								k = getKeyIndex curObj.rotation.controller (curChn.lastFrame + 1)
								if k == 0 then
								(
									curKey = addNewKey curObj.rotation.controller (curChn.lastFrame + 1)
									curKey.value = datumRot
								)
							)
							else
							(
								k = getKeyIndex curObj.rotation.controller (curChn.firstFrame - 1)
								if k == 0 then
								(
									curKey = addNewKey curObj.rotation.controller (curChn.firstFrame - 1)
									curKey.value = datumRot
								)
							)
						)
					)
				) --for end
			) --if end

			for i = 1 to anim.bitchannels.count do
			(
				curChn = anim.bitchannels[i]
				-- curObj = execute ("$'" + hier.pivots[curChn.pivotID].pivotName + "'")
				-- at v1.02. it should be curChn.pivotID + 1 at v1.04
				curObj = gmPivots[curChn.pivotID] -- test at v1.04
				if curObj != undefined then
				(
					case curChn.flags of
					(
						0x0000: --BIT_CHANNEL_VIS = 0	// turn meshes on and off depending on anim frame.
						(
							defVal = (curChn.defaultVal > 0)
							curObj.visibility = defVal
							curObj.visibility.controller = On_Off()
							curKey = addNewKey curObj.visibility.controller 0
							curKey.selected = defVal
							prevVal = defVal
							byteIdx = 1
							bitIdx = 1
							for f = curChn.firstFrame to curChn.lastFrame do
							(
								curVal = bit.get (curChn.values[byteIdx]) bitIdx
								if curVal != prevVal then
								(
									curKey = addNewKey curObj.visibility.controller (f + 1)
									curKey.selected = curVal
									prevVal = curVal
								)
								bitIdx += 1
								if bitIdx > 8 then
								(
									byteIdx += 1
									bitIdx = 1
								)
							)
							curKey = addNewKey curObj.visibility.controller (curChn.lastFrame + 1)
							curKey.selected = defVal
						)
						--0x0001: --BIT_CHANNEL_TIMECODED_VIS
					)
				)
			) --for end
		) --if end

		-----------------------------------------------------------------------------
		-- TASK: Create model animations typeof TIME-CODED + COMPRESSED ....
		-----------------------------------------------------------------------------
		if (cmpAnim != undefined) and (hier != undefined) then
		(
			if cmpAnim.header.frameCount > 1 then
				animationRange = interval 1 cmpAnim.header.frameCount -- edited at v1.07
			else
				animationRange = interval 0 1
			frameRate = cmpAnim.header.frameRate
			case cmpAnim.header.flavor of
			(
				0x0: --ANIM_FLAVOR_TIMECODED
				(
					for i = 1 to cmpAnim.channels.count do
					(
						curChn = cmpAnim.channels[i]
						curObj = gmPivots[curChn.pivotID] -- test at v1.04
						if curObj != undefined then
						(
							datumPos = hier.pivots[curChn.pivotID + 1].pos
							datumRot = hier.pivots[curChn.pivotID + 1].rotation
							case curChn.flags of
							(
								0x0000: --ANIM_CHANNEL_TIMECODED_X = 0
								(
									curObj.pos.controller = linear_position()
									curKey = addNewKey curObj.pos.controller 0
									curKey.value = datumPos
									for t = 1 to curChn.timeCodesCount do
									(
										fNext = curChn.values[t].keyTime
										vNext = curChn.values[t].keyValue
										if t > 1 then
										(
											fPrev = curChn.values[(t - 1)].keyTime + 1
											vPrev = curChn.values[(t - 1)].keyValue
										)
										else
										(
											fPrev = fNext
											vPrev = vNext
										)
										for f = fPrev to fNext do
										(
											k = getKeyIndex curObj.pos.controller (f + 1)
											if k == 0 then
											(
												curKey = addNewKey curObj.pos.controller (f + 1)
												curKey.value = datumPos
											)
											else
											(
												curKey = curObj.pos.controller.keys[k]
											)
											step = vPrev + ((vNext - vPrev) * (f - fPrev + 1) / (fNext - fPrev + 1))
											curKey.value += [step, 0, 0] * (inverse datumRot)
										)
									)
								)
								0x0001: --ANIM_CHANNEL_TIMECODED_Y = 1
								(
									curObj.pos.controller = linear_position()
									curKey = addNewKey curObj.pos.controller 0
									curKey.value = datumPos
									for t = 1 to curChn.timeCodesCount do
									(
										fNext = curChn.values[t].keyTime
										vNext = curChn.values[t].keyValue
										if t > 1 then
										(
											fPrev = curChn.values[(t - 1)].keyTime + 1
											vPrev = curChn.values[(t - 1)].keyValue
										)
										else
										(
											fPrev = fNext
											vPrev = vNext
										)
										for f = fPrev to fNext do
										(
											k = getKeyIndex curObj.pos.controller (f + 1)
											if k == 0 then
											(
												curKey = addNewKey curObj.pos.controller (f + 1)
												curKey.value = datumPos
											)
											else
											(
												curKey = curObj.pos.controller.keys[k]
											)
											step = vPrev + ((vNext - vPrev) * (f - fPrev + 1) / (fNext - fPrev + 1))
											curKey.value += [0, step, 0] * (inverse datumRot)
										)
									)
								)
								0x0002: --ANIM_CHANNEL_TIMECODED_Z = 2
								(
									curObj.pos.controller = linear_position()
									curKey = addNewKey curObj.pos.controller 0
									curKey.value = datumPos
									for t = 1 to curChn.timeCodesCount do
									(
										fNext = curChn.values[t].keyTime
										vNext = curChn.values[t].keyValue
										if t > 1 then
										(
											fPrev = curChn.values[(t - 1)].keyTime + 1
											vPrev = curChn.values[(t - 1)].keyValue
										)
										else
										(
											fPrev = fNext
											vPrev = vNext
										)
										for f = fPrev to fNext do
										(
											k = getKeyIndex curObj.pos.controller (f + 1)
											if k == 0 then
											(
												curKey = addNewKey curObj.pos.controller (f + 1)
												curKey.value = datumPos
											)
											else
											(
												curKey = curObj.pos.controller.keys[k]
											)
											step = vPrev + ((vNext - vPrev) * (f - fPrev + 1) / (fNext - fPrev + 1))
											curKey.value += [0, 0, step] * (inverse datumRot)
										)
									)
								)
								0x0006: --ANIM_CHANNEL_TIMECODED_Q = 6
								(
									curObj.rotation.controller = linear_rotation()
									curKey = addNewKey curObj.rotation.controller 0
									curKey.value = datumRot
									for t = 1 to curChn.timeCodesCount do
									(
										f = curChn.values[t].keyTime
										curKey = addNewKey curObj.rotation.controller (f + 1)
										curKey.value = curChn.values[t].keyValue - (inverse datumRot)
									)
								)
							)
						)
					)

					for i = 1 to cmpAnim.bitchannels.count do
					(
						curChn = cmpAnim.bitchannels[i]
						curObj = gmPivots[curChn.pivotID] -- test at v1.04
						if curObj != undefined then
						(
							case curChn.flags of
							(
								--0x0000: --BIT_CHANNEL_VIS = 0	// turn meshes on and off depending on anim frame.
								0x0001: --BIT_CHANNEL_TIMECODED_VIS = 1
								(
									defVal = (curChn.defaultVal > 0)
									curObj.visibility = defVal
									curObj.visibility.controller = On_Off()
									curKey = addNewKey curObj.visibility.controller 0
									curKey.selected = defVal
									prevVal = defVal
									byteIdx = 1
									bitIdx = 1
									for t = 1 to curChn.timeCodesCount do
									(
										f = curChn.values[t].time
										curVal = bit.get (curChn.values[byteIdx]) bitIdx
										if curVal != prevVal then
										(
											curKey = addNewKey curObj.visibility.controller (f + 1)
											curKey.selected = curVal
											prevVal = curVal
										)
										bitIdx += 1
										if bitIdx > 8 then
										(
											byteIdx += 1
											bitIdx = 1
										)
									)
								)
							)
						)
					) --for end
				)
				--0x1: --ANIM_FLAVOR_ADAPTIVE_DELTA
				--0x2: --ANIM_FLAVOR_VALID
			)
		) --if end

		-------------------------------------------
		if pickbox != undefined then
		(
			sName = pickbox.boxName
			dotPos = findString sName "."
			if (dotPos != undefined) and (dotPos < sName.count) do (sName = subString sName (dotPos + 1) -1)
			gmBox = Box name:sName pos:pickbox.center
			gmBox.width = pickbox.extent.x
			gmBox.length = pickbox.extent.y
			gmBox.height = pickbox.extent.z
			gmBox.wirecolor = pickbox.boxColor
		)
	)
)

-----------------------------------------------------
-------------------------------------- Main Entry ---
macroscript ImporterW3D
	category: "LOTR ModelW3D Utility"
	buttontext: "Import W3D (BFME)"
	tooltip: "W3D Importer for BFME - Coolfile"
	icon:#("gMax",2)
(
	cfW3DImporter()
)
-- ***********************************************************
-- Module:  w3DImporter.ms -- End of File
-- ***********************************************************
