-- ************************************************************************
-- Project: Modding Tools & Utilities
-- ------------------------------------------------------------------------
-- Purpose: Basic methods and members to bind and link W3D mmodel
-- Module : W3DMeshHelp.ms
-- Author : Level_3D_Vision (c) 2008-2011
-- System : MaxScript utility for GMAX
-- ------------------------------------------------------------------------
-- History: 18.11.09 -- created by Jerry Brutzki
--          22.08.11 -- changed by Jerry Brutzki
-- ************************************************************************
------------------------------------------------------------- Variables ---
---------------------------------------------------------------------------
ModelList = #(); ModelName = #(); PivotList = #(); sceneOBJ = #()
PivotLink = #(); PivotSave = #(); PivotName = #()
meshIdx = 0
boneIdx = 0
linkIdx = 0
vertPos = 0.0

---------------------------------------------------------------------------
------------------------------------------------------------ structures ---
struct FileReader
---------------------------------------------------------------------------
(
    headLine = "Import W3D VertexLinks (BFME)",
    userType = "VertexLinks W3D (*.txt)|*.txt|All Files (*.*)|*.*|",
    fileName = undefined, fStream  = undefined,

    fn openFileInput =
    (
       fileName = getOpenFileName caption:headLine types:userType
       if (fileName != undefined) then
       (
          fStream = openFile fileName
          return fStream
       )
       return undefined
    ),

    fn GetNextLine =
    (
       return readLine fStream
    ),

    fn closeFileInput =
    (
       if (fStream != undefined) then
       (
          close fStream
       )
    )
)

------------------------------------------------------------- Functions ---
---------------------------------------------------------------------------
function ShowMessage userText =
---------------------------------------------------------------------------
(
    MessageBox userText title:"Model Helper Message"
)

---------------------------------------------------------------------------
function ObtainModelMesh =
---------------------------------------------------------------------------
(
    PivotName = #(); PivotList =#(); PivotLink = #()
    clearSelection();    select objects
    sceneOBJ = getCurrentSelection()
    deselect objects

    if (sceneOBJ.count == 0) then
    (
       ShowMessage ("No imported W3D model found ...")
    )

    if (sceneOBJ.count > 0) then
    (
       for obj in sceneOBJ do
       (
          if ((classOf obj) == Editable_Mesh) then
          (
             append ModelName obj.name
             append ModelList obj
          )

          if ((classOf obj) == Box) then
          (
             append ModelName obj.name
             append ModelList obj
          )

          if ((classOf obj) == Bone) then
          (
             append PivotName obj.name
             append PivotList obj
             --- append PivotLink "0...0"
          )
       )
    )
    clearSelection()
    sceneOBJ = #()
)

---------------------------------------------------------------------------
function WeldMeshVertices currIdx currPos =
---------------------------------------------------------------------------
(
   local vertList = #()
   local nodeList = #()
   local weldList = #()
   local showList = #()
   local offset = 0.0
   local margin = 0.2
   

   if (currIdx > 0) then
   (
      if (currPos > 0.0) then (
         margin = currPos
      )
      format "MESHNAME: % Margin: [%]\n" ModelList[currIdx].name margin
      format "==========================================================\n"

      clearSelection()
      select ModelList[currIdx]
      -- Deletes all vertices from model not used by a face .... 
      meshop.deleteIsoVerts ModelList[currIdx]

      showList = meshop.getHiddenVerts ModelList[currIdx]
      for showVTX in showList do
      (
         format "Obscured VTX: [%]\n" showVTX.pos
         deleteVert ModelList[currIdx] showVTX.index 
      )

      select ModelList[currIdx].verts
      vertList = ModelList[currIdx].selectedVerts
      nodeList = ModelList[currIdx].selectedVerts
      for currVTX in vertList do
      (
         for nodeVTX in nodeList do
         (
            offset = distance currVTX.pos nodeVTX.pos
            if (offset <= margin) then
            (
               if (offset >= 0) then
               (
                  format "Distance: [%][%] <= [%][%]\n" currVTX.index \
                          currVTX.pos nodeVTX.index nodeVTX.pos
                  append weldList currVtx.index
                  append weldList nodeVtx.index
                  exit()
                )
            )
         )
      )

      -- Welds the specified vertices that are within the threshold distance.
      if (weldList.count > 0) then
      (
         meshop.weldVertsByThreshold ModelList[currIdx] weldList margin 
      )

      deselect ModelList[currIdx].verts
      update ModelList[currIdx]
   )
)

---------------------------------------------------------------------------
function SizeMeshVertices currIdx =
---------------------------------------------------------------------------
(
   if (currIdx > 0) then
   (
      select ModelList[currIdx]
      scale ModelList[currIdx] [0.877193,0.877193,0.877193]
      update ModelList[currIdx]

      max scale
      max tool maximize
   )
)

---------------------------------------------------------------------------
function ImportPivotLinks MeshLinkList =
---------------------------------------------------------------------------
(
    local FReader  = FileReader()
    local FStream  = FReader.openFileInput()
    local scanLine = #()
    local userLink = #()

    if (FStream != undefined) then
    (
       while ((eof FStream) == false) do
       (
          currLine = FReader.GetNextLine()
          if (currLine != undefinded) then
          (
             scanLine  = filterString currLine "[]"
             if (scanLine[1] == "Vertex_Link: ") Then
             (
                lftNumber = substring scanLine[2] 1 4
                rhtNumber = substring scanLine[2] 5 9
                append userLink ((trimLeft lftNumber) + "," + (trimLeft rhtNumber))
             )
          )
       )
    )
    FReader.closeFileInput()

    if (userLink.count > 0) then
    (
       MeshLinkList.items = userLink
    )
)

---------------------------------------------------------------------------
function SelectVertexLinks currValue =
---------------------------------------------------------------------------
(
    local tickVertices = #()
    local vertLinks = #()

    if (currValue != "0...0") then
    (
       fromVtx = 0; lastVtx = 0
       vertLinks = filterString currValue ","
       if (vertLinks.count == 2) then
       (
          fromVtx = (vertLinks[1] as integer)
          lastVtx = (vertLinks[2] as integer)
       )

       try (
          clearSelection()
          select ModelList[meshIdx]
          max modify mode; subObjectLevel = 1

          for nextVtx = fromVtx to lastVtx do
          (
             append tickVertices nextVtx
          )

          if ((classOf ModelList[meshIdx]) == Editable_Mesh) then
          (
             --- ModelList[meshIdx].selectedVerts = tickVertices
             setVertSelection ModelList[meshIdx] tickVertices
             update ModelList[meshIdx]
             modPanel.setCurrentObject ModelList[meshIdx]
          )
       )
       catch
       (
          ShowMessage ("Mesh Object missing - Check your scene ...")
       )
   )
)

---------------------------------------------------------------------------
function ClearAllMeshData =
---------------------------------------------------------------------------
(
    ModelList = #(); ModelName = #(); PivotList = #(); sceneOBJ = #()
    PivotLink = #(); PivotSave = #(); PivotName = #()
    meshIdx = 0
    boneIdx = 0
    linkIdx = 0
    vertPos = 0.0
    clearSelection()
)

---------------------------------------------------------------------------
rollout MeshLinkPanel "W3D Model Helper"
---------------------------------------------------------------------------
(
    listbox listMesh pos:[5,6] width:150 height:8
    button  loadMesh "Load Model" pos:[163,6] width:70 height:18
    button  loadVert "Load Links" pos:[163,30] width:70 height:18
    button  cropMesh "Crop Model" pos:[163,54] width:70 height:18
    spinner useRange ">>> " pos:[163,78] width:70 height:18 fieldwidth:28 range:[0,1.2,0] type:#float
    button  sizeMesh "Size Model" pos:[163,100] width:70 height:18

    button  userInfo "Summary Model" pos:[5,127] width:150 height:18
    button  userExit "Close" pos:[163,127] width:70 height:18

    listbox listBone pos:[5,152] width:150 height:22
    listBox meshLink pos:[163,152] width:70 height:22

    --- Event handler: Import and Display ---
    on loadMesh pressed do
    (
       ClearAllMeshData()
       ObtainModelMesh()
       if (ModelList.count > 0) then
       (
          listMesh.items = #()
          listMesh.items = ModelName
       )

       if (PivotList.count > 0) then
       (
          listBone.items = #()
          listBone.items = PivotName
          meshLink.items = #()
       )
    )

    on listMesh selected listIdx do
    (
       try (
         meshIdx = listIdx
         select ModelList[meshIdx]
       )
       catch
       (
          ShowMessage ("Mesh Object missing - Check your scene ...")
       )
    )
    on listMesh doubleClicked listIdx do
    (
       try (
          meshIdx = listIdx
          select ModelList[meshIdx]
          max properties
       )
       catch
       (
          ShowMessage ("Mesh Object missing - Check your scene ...")
       )
    )

    on cropMesh pressed do
    (
       if (ModelList.count > 0) then
       (
          WeldMeshVertices meshIdx vertPos
       )
    )

    on sizeMesh pressed do
    (
       if (ModelList.count > 0) then
       (
          SizeMeshVertices meshIdx
       )
    )

    on useRange changed currVal do
    (
       vertPos = currVal
    )
    on useRange entered do
    (
       vertPos = useRange.value
    )

    on loadVert pressed do
    (
       if (ModelList.count > 0) then
       (
          ImportPivotLinks meshLink
       )
    )

    --- Event handler: Select and Display infos ---
    on listBone selected userIdx do
    (
       try (
          boneIdx = userIdx
          select PivotList[boneIdx]
       )
       catch
       (
          ShowMessage ("Bone Object missing - Check your scene ...")
       )
    )
    on listBone doubleClicked userIdx do
    (
       try (
          boneIdx = userIdx
          select PivotList[boneIdx]
          max properties
       )
       catch
       (
          ShowMessage ("Bone Object missing - Check your scene ...")
       )
    )
    on meshLink selected userIdx do
    (
       if (meshLink.selected != undefined) then
       (
          linkIdx = userIdx
          SelectVertexLinks meshLink.selected
       )
    )
    on meshLink doubleClicked userIdx do
    (
       if (meshLink.selected != undefined) then
       (
          linkIdx = userIdx
          SelectVertexLinks meshLink.selected
       )
    )

    on userInfo pressed do
    (
       if (ModelList.count > 0) then
       (
          max file summary
       )
    )

    --- Event handler: Finish and closing ---
    on userExit pressed do
    (
       if (ModelList.count > 0) then
       (
          listMesh.items = #()
          listBone.items = #()
          meshLink.items = #()
          ClearAllMeshData()
       )
       DestroyDialog MeshLinkPanel
    )
)

---------------------------------------------------------------------------
------------------------------------------------------------ ScriptName ---
macroscript W3DMeshLinks
    category: "LOTR ModelW3D Utility"
    buttontext: "BFME Model Helper"
    tooltip: "Mesh and Pivot information"
    icon:#("Maintoolbar",9)
(
    settings = #(#style_titlebar,#style_border,#style_sysmenu,#style_minimizebox)
    CreateDialog MeshLinkPanel width:238 height:446 style:settings
)

-- ************************************************************************
-- Module:  W3DMeshHelp.ms -- End of File
-- ************************************************************************
